/**
  ******************************************************************************
  * @file    cp_server.c 
  * @author  Ruediger R. Asche
  * @version V1.0.0
  * @date    July 14, 2016
  * @brief   connection establishment code, server side
  ******************************************************************************
  * @attention
  *
  * THE PRESENT FIRMWARE WHICH IS FOR GUIDANCE ONLY AIMS AT PROVIDING CUSTOMERS
  * WITH CODING INFORMATION REGARDING THEIR PRODUCTS IN ORDER FOR THEM TO SAVE
  * TIME. AS A RESULT, THE AUTHOR SHALL NOT BE HELD LIABLE FOR ANY
  * DIRECT, INDIRECT OR CONSEQUENTIAL DAMAGES WITH RESPECT TO ANY CLAIMS ARISING
  * FROM THE CONTENT OF SUCH FIRMWARE AND/OR THE USE MADE BY CUSTOMERS OF THE
  * CODING INFORMATION CONTAINED HEREIN IN CONNECTION WITH THEIR PRODUCTS.
  ******************************************************************************  
  */ 

#include "project.h"

#include "cp.h"

/** @brief Manages the server side control flow of the communication 
 *
 * 
 *  @param p_LocalPort local TCP endpoint of a communication
 *  @param p_Processor communication processor
 *
 *  @return status code (see CP_STATUSCODE_XXX enumerator in cp_frame.h)
 *
 * This logic implements a serialized TCP server, meaning that the server only
 * serves one client connection at any time and can only serve another client
 * when the current client has terminated its connection. 
 */

CP_STATUSCODE cp_CommLoopServer(unsigned short p_LocalPort,PCP_PROCESSOR p_Processor)
{
    CP_STATUSCODE a_Return = CP_STATUSCODE_FATAL; // some problem in the network layer (eg port double bound)
    int a_ListenerSocket=socket(AF_INET,SOCK_STREAM,0);
    if (p_Processor && a_ListenerSocket >= 0)
    {
        struct sockaddr_in a_Address;
        a_Address.sin_family = AF_INET;
        a_Address.sin_port = htons(p_LocalPort);
        a_Address.sin_addr.s_addr = INADDR_ANY;
        if (bind(a_ListenerSocket,(struct sockaddr *)&a_Address, sizeof(struct sockaddr_in)) >= 0)
        {
            if (listen(a_ListenerSocket,1) >= 0)
            {
                do
                {
                    struct sockaddr_in a_PeerAddress;
                    socklen_t a_PeerSize=sizeof(a_PeerAddress);
                    int a_WorkerSocket = accept(a_ListenerSocket,(struct sockaddr *)&a_PeerAddress,&a_PeerSize);
                    a_Return = CP_STATUSCODE_FATAL; // reinitialize for this iteration
                    // TODO: There are scenarios in which accept() may return in a non-fatal fashion (eg a client connects and
                    // immediately disconnects). Make sure that these cases (detectable by the error code in the socket) do not
                    // cause this loop to terminate!                    
                    if (a_WorkerSocket >= 0)
                    {
                        int a_timeout = 100; /* 1000 msecs */
                        if (setsockopt(a_WorkerSocket, SOL_SOCKET, SO_RCVTIMEO, &a_timeout, sizeof(a_timeout)) == 0)
                        {
                            p_Processor->m_Socket = a_WorkerSocket;
                            // signal the receiver task that the socket to read from is now valid
                            xTaskNotifyGive( p_Processor->m_AssociatedReceiverTask );
                            cp_QueuePacketToComm(p_Processor,TAG_AUTHREQUEST,CP_MAGIC_NUMBER_AUTHENTICATION,CP_PRIQUEUE_MEDIUMPRI);                            
                            a_Return = cp_HandleEstablishedComm(p_Processor);                          
                        } // could set a timeout
                        shutdown(a_WorkerSocket,SHUT_RDWR);
                        close(a_WorkerSocket);
                    } // a client connected
                } // sequential server loop
                while (a_Return != CP_STATUSCODE_FATAL);
            } // listen() succeeded
        } // bind() succeeded
        close(a_ListenerSocket);        
    } // could create listener socket
    // on the server side, we only get here if something is badly broken!
    return a_Return;
}
